%% Step 1: Generate WindTrax Inputs - Specific Degree Folders
clear all;
close all;

% Define the parent directory containing the folders
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables\MGGA\Emission Sources\Multi_Point\Emission Point\Quantified_MRSP';
minute_folders = {'5 minutes', '15 minutes', '30 minutes'};
degree_folders_to_process = {'5 degrees', '10 degrees', '20 degrees'}; % Specific degree folders to process

% Loop through each minute folder
for m = 1:numel(minute_folders)
    folderName = minute_folders{m};
    minutePath = fullfile(parentPath, folderName);
    
    % Loop through each specified degree folder
    for d = 1:numel(degree_folders_to_process)
        degreeFolder = degree_folders_to_process{d};
        degreePath = fullfile(minutePath, degreeFolder);
        
        % Check if the degree folder exists
        if ~exist(degreePath, 'dir')
            warning('Degree folder not found: %s', degreePath);
            continue;
        end

        % Create a new "WindTrax" folder inside the degreePath folder
        windTraxPath = fullfile(degreePath, 'WindTrax');
        if ~exist(windTraxPath, 'dir')
            mkdir(windTraxPath);
        end

        % Get a list of all CSV files in the Single_Emission folder
        files = dir(fullfile(degreePath, '*.csv'));
        
        % Check if any files were found
        if isempty(files)
            warning('No CSV files found in: %s', fullfile(degreePath));
            continue;
        end

        % Process each CSV file
        for k = 1:numel(files)
            filename = files(k).name;
            filePath = fullfile(degreePath, filename);
            data = readtable(filePath);

            % Convert pressure from hPa to Pa
            data.P_Pa = data.BP * 100;

            % Calculate distance from the emission point to the met & concentration sensor
            % MGGA latitude and longitude (0,0) WindTrax coordinate

            % Convert aeris latitude and longitude to radians
            source_lat_rad = deg2rad(data.MGGA_lat_N);
            source_lon_rad = deg2rad(data.MGGA_long_W);
            
            % Convert the latitudes and longitudes in data to radians
            latitudes_rad = deg2rad(data.Lat);
            longitudes_rad = deg2rad(data.Lon);
            
            % Radius of the Earth in m
            earth_radius_m = 6371000;  % ~6371 km * 3280.84 feet/km
            
            % Calculate the differences in latitude and longitude
            delta_lat = latitudes_rad - source_lat_rad;
            delta_lon = longitudes_rad - source_lon_rad;
            
            % Calculate x and y distances
            x_distance = earth_radius_m * delta_lon .* cos(source_lat_rad);  % East-West distance
            y_distance = earth_radius_m * delta_lat;                        % North-South distance
            
            % Add the x and y distances to the data table
            data.x_distance_m = x_distance;
            data.y_distance_m = y_distance;

            % Save the updated table into the "WindTrax" folder
            outputFile = fullfile(windTraxPath, filename);
            writetable(data, outputFile);
            
            fprintf('Processed: %s\n', outputFile);
        end
    end
end

disp('Processing and file saving completed for specified degree folders.');

%% Step 2: Select 1st rows of all tables in a folder and combine them

clear all;
close all;

% Define the parent directory containing the folders
parentPath = '\\csunas01.colostate.edu\research\metec\user\mbua\ADED\OTM33_Gaussian\Event Tables\MGGA\Emission Sources\Multi_Point\Emission Point\Quantified_MRSP';
outputPath = 'C:\Users\mbua\Desktop\ADED\WindTrax\Multi_Point_Single_Emission';

% Create the output directory if it doesn't exist
if ~exist(outputPath, 'dir')
    mkdir(outputPath);
end

% Specify the minute and stability folders
minute_folders = { '30 minutes'};

% Loop through each minute folder
for m = 1:numel(minute_folders)
    folderName = minute_folders{m};
    minutePath = fullfile(parentPath, folderName);
    
    % Get a list of all "* degrees" subfolders
    degreeFolders = dir(fullfile(minutePath, '* degrees'));
    degreeFolders = degreeFolders([degreeFolders.isdir]); % Filter directories only

    % Loop through each "* degrees" folder
    for d = 1:numel(degreeFolders)
        degreeFolder = degreeFolders(d).name;
        degreePath = fullfile(minutePath, degreeFolder);

        % Check if the WindTrax folder exists
        filesPath = fullfile(degreePath,'WindTrax');
        if ~exist(filesPath, 'dir')
            warning('folder %s does not exist. Skipping...', filesPath);
            continue;
        end

        % Get a list of all CSV files in the WindTrax folder
        files = dir(fullfile(filesPath, '*.csv'));
        if isempty(files)
            warning('No CSV files found in %s. Skipping...', filesPath);
            continue;
        end

        % Initialize a table to store the combined rows
        combinedTable = table();

        % Process each CSV file in the folder
        for k = 1:numel(files)
            filename = files(k).name;
            filePath = fullfile(filesPath, filename);

            % Read the table and extract the first row
            data = readtable(filePath);
            firstRow = data(1, :); % Extract the first row
            combinedTable = [combinedTable; firstRow]; % Append to the combined table
        end

        % Determine the output file name based on the file path
        outputFileName = sprintf('%s_%s_WindTrax.csv', folderName, degreeFolder);
        outputFilePath = fullfile(outputPath, outputFileName);

        % Write the combined table to the output path
        writetable(combinedTable, outputFilePath, 'FileType', 'text', 'Delimiter', ',');
    end
end

disp('Combination and file saving completed.');
